# Network Visualization Guide

## Quick Start Examples

### Loading Data in R
```r
# Install required packages
install.packages(c("igraph", "tidygraph", "ggraph"))

# Load GraphML format (recommended)
library(igraph)
g <- read_graph("waipahu/03-graphml/waipahu.graphml", format = "graphml")

# Basic visualization
plot(g, vertex.size=5, vertex.label=NA, edge.arrow.size=0.3)

# Advanced visualization with ggraph
library(tidygraph)
library(ggraph)

tidy_g <- as_tbl_graph(g)
ggraph(tidy_g) +
  geom_edge_link(arrow = arrow(length = unit(2, 'mm'))) +
  geom_node_point(aes(color = role), size = 3) +
  theme_graph()
```

### Loading Data in Python
```python
import networkx as nx
import pandas as pd
import matplotlib.pyplot as plt

# Load GraphML format
G = nx.read_graphml("waipahu/03-graphml/waipahu.graphml")

# Basic visualization
pos = nx.spring_layout(G)
nx.draw(G, pos, with_labels=False, node_size=50,
        arrows=True, arrowsize=20)
plt.show()

# Load CSV for pandas analysis
edges = pd.read_csv("waipahu/02-csv/edges.csv")
nodes = pd.read_csv("waipahu/02-csv/nodes.csv")

# Create network from CSV
G_csv = nx.from_pandas_edgelist(edges, source='source', target='target',
                                create_using=nx.DiGraph())
```

### Opening in Gephi
1. Download and install Gephi (https://gephi.org/)
2. File → Open → Select the .gexf file
3. Choose "Directed" graph type
4. Apply layout algorithms (ForceAtlas2 recommended)
5. Color nodes by role: Statistics → Average Degree
6. Export publication-quality images

### Opening in Cytoscape
1. Download Cytoscape (https://cytoscape.org/)
2. File → Import → Network from File
3. Select GraphML or CSV edge file
4. Apply layout: Layout → Prefuse Force Directed
5. Style networks by attributes in the Style panel

## Analysis Recommendations

### Network Metrics to Calculate
- **Degree centrality:** Identifies highly connected individuals
- **Betweenness centrality:** Finds bridge connectors
- **Clustering coefficient:** Measures local connectivity
- **Density:** Overall network connectivity
- **Reciprocity:** Bidirectional relationship strength

### Temporal Analysis
- Compare networks across different time periods
- Track centrality changes over observation dates
- Analyze interaction patterns by hour/day

### Spatial Analysis
- Examine room-based interaction patterns
- Identify spaces that facilitate connections
- Compare interaction types by location

### Role-Based Analysis
- Staff vs. Member interaction patterns
- Guest integration into community
- Role transitions and network changes

## Common Issues & Solutions

**Large Networks:** Use filtered data or layout algorithms designed for large graphs
**Missing Data:** Check for null values in CSV files before analysis
**Directed vs Undirected:** Most data is directional (A → B), choose appropriate algorithms
**Temporal Overlays:** Use timestamp data to create dynamic visualizations
**Publication Graphics:** GEXF → Gephi provides best publication-quality outputs

## Research Questions This Data Can Address

1. **Social Integration:** How do new members connect to existing networks?
2. **Staff Roles:** What bridging functions do staff members serve?
3. **Spatial Effects:** How do physical spaces influence social connections?
4. **Engagement Quality:** Relationship between interaction valence and network position
5. **Temporal Patterns:** How do social networks evolve throughout observation periods?

For additional analysis examples, see the research documentation or contact the research team.